#include "SMManager.h"
#include "boost/date_time/posix_time/posix_time.hpp"
#include <iostream>
using namespace boost;

TSMManager::TSMManager(  const long long _floatHistory, TVoidAllocator _void_alloc)
	: void_alloc( _void_alloc), newestTimestamp( -1), floatHistory( _floatHistory), imageQueues( _void_alloc), queueRegisterCounts( _void_alloc), idMap( std::less< TCharString>(), _void_alloc),
	floatQueues( _void_alloc), floatQueueRegisterCounts( _void_alloc), idFloatMap( std::less< TCharString>(), _void_alloc)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	imageQueues.reserve( 100);
	queueRegisterCounts.reserve( 100);
}

TSMManager::~TSMManager()
{
}


void TSMManager::registerFloatQueue( const std::string &queueName)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	TCharString name( queueName.data(), void_alloc);

	TIDMap::const_iterator idIt = idFloatMap.find( name);

	if( idIt == idFloatMap.end()){
		idFloatMap[ name] = floatQueueRegisterCounts.size();
		floatQueues.push_back( TFloatQueue( std::less<long long>(), void_alloc) );
		floatQueueRegisterCounts.push_back( 1);
	}

	change();
}

void TSMManager::deregisterFloatQueue( const std::string &queueName)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	floatQueueRegisterCounts[ idFloatMap[ TCharString( queueName.data(), void_alloc) ] ]--;

	change();
}

void TSMManager::getActiveFloatQueues( std::map< std::string, bool> &activeQueues)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	activeQueues.clear();

	for( TIDMap::const_iterator idIt = idFloatMap.begin(); idIt != idFloatMap.end(); idIt++){
		activeQueues[ idIt->first.data()] = floatQueueRegisterCounts[ idIt->second] > 0;
	}
}

void TSMManager::push( const std::string &queueName, const std::vector< float> &vec, const long long timestamp)
{

	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	TIDMap::const_iterator idIt = idFloatMap.find( name);
	if( idIt == idFloatMap.end()){
		throw( "TSMManager::push(): The requested vector< float> queue does not exist.");
	}

	// update newest timestamp
	newestTimestamp = std::max( newestTimestamp, timestamp);

	// add the vector in the queue
	floatQueues[ idIt->second].insert( TFloatQueueValue( timestamp, TFloatVector( vec.begin(), vec.end(), void_alloc)));

	change();
}

void TSMManager::get( const std::string &queueName, const long long timestamp, std::vector< std::vector< float> > &result)
{
	result.clear();
	result.reserve( 100);

	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idFloatMap.find( name);
	if( idIt == idFloatMap.end()){
		throw( "TSMManager::get(): The requested queue does not exist.");
	}

	const int id = idIt->second;

	// get the vectors with the timestamp
	std::pair< TFloatQueue::iterator, TFloatQueue::iterator> range = floatQueues[ id].equal_range( timestamp);

	// copy vectors with the requested timestamp
	for( TFloatQueue::iterator rIt = range.first; rIt != range.second; rIt++){
		result.push_back( std::vector< float>( rIt->second.begin(), rIt->second.end()));
	}
}

void TSMManager::getNewest( const std::string &queueName, long long &timestamp, std::vector< std::vector< float> > &result)
{
	result.clear();
	result.reserve( 100);

	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idFloatMap.find( name);
	if( idIt == idFloatMap.end()){
		throw( "TSMManager::getImage(): The requested queue does not exist.");
	}

	const int id = idIt->second;

	// get the vectors with the timestamp
	TFloatQueue::reverse_iterator newIt = floatQueues[ id].rbegin();

	if( newIt != floatQueues[ id].rend()){
		for( TFloatQueue::reverse_iterator it = newIt; it->first == newIt->first; it++){
			result.push_back( std::vector< float>( it->second.begin(), it->second.end()));
		}
	}
}
	

void TSMManager::getNewer( const std::string &queueName, const long long timestamp, std::multimap< long long, std::vector< float> > &result)
{
	result.clear();

	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idFloatMap.find( name);
	if( idIt == idFloatMap.end()){
		throw( "TSMManager::get(): The requested queue does not exist.");
	}

	const int id = idIt->second;

	// copy vectors with the requested timestamp
	for( TFloatQueue::iterator rIt = floatQueues[ id].lower_bound( timestamp); rIt != floatQueues[ id].end(); rIt++){
		result.insert( result.end(), std::pair< long long, std::vector< float> >( rIt->first, std::vector< float>( rIt->second.begin(), rIt->second.end())));
	}
}









void TSMManager::registerQueue( const std::string &queueName)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	TCharString name( queueName.data(), void_alloc);

	TIDMap::const_iterator idIt = idMap.find( name);

	if( idIt == idMap.end()){
		idMap[ name] = queueRegisterCounts.size();
		imageQueues.push_back( TImageQueue( std::less<long long>(), void_alloc) );
		queueRegisterCounts.push_back( 1);
	}

	change();
}


void TSMManager::deregisterQueue( const std::string &queueName)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	queueRegisterCounts[ idMap[ TCharString( queueName.data(), void_alloc) ] ]--;

	change();
}

void TSMManager::getActiveQueues( std::map< std::string, bool> &activeQueues)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	activeQueues.clear();

	for( TIDMap::const_iterator idIt = idMap.begin(); idIt != idMap.end(); idIt++){
		activeQueues[ idIt->first.data()] = queueRegisterCounts[ idIt->second] > 0;
	}
}

void TSMManager::pushImage( const std::string &queueName, TSMImage *img, const long long timestamp)
{

	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	TIDMap::const_iterator idIt = idMap.find( name);
	if( idIt == idMap.end()){
		throw( "TSMManager::pushImage(): The requested queue does not exist.");
	}

	TImageQueue::const_iterator tsIt = imageQueues[ idIt->second].find( timestamp);

	// check if an image with this timestamp is alredy in the queue
	if( tsIt != imageQueues[ idIt->second].end() && imageQueues[ idIt->second][  timestamp].imgPtr != NULL){
		throw( "TSMManager::pushImage(): Trying to push image with timestamp which is already present in the corresponding queue;");
	}

	newestTimestamp = std::max( newestTimestamp, timestamp);

	// add the image in the queue while perserving the useCount which could have been already changed

	imageQueues[ idIt->second][  timestamp].imgPtr = img;

	change();
}


long long TSMManager::getNewest( const std::string &queueName)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idMap.find( name);
	if( idIt == idMap.end()){
		throw( "TSMManager::getNewestImage(): The requested queue does not exist.");
	}

	// check if there are any images in the requested queue
	if( imageQueues[ idIt->second].empty()){
		return -1;
	}

	// get the image iterator
	TImageQueue::reverse_iterator imgIt = imageQueues[ idIt->second].rbegin();

	while( imgIt != imageQueues[ idIt->second].rend()){

		if( imgIt->second.imgPtr != NULL){
			return imgIt->first;
		}

		imgIt++;
	}

	return -1;
}

void TSMManager::registerImage( const std::string &queueName, long long timestamp)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idMap.find( name);
	if( idIt == idMap.end()){
		throw( "TSMManager::registerImage(): The requested queue does not exist.");
	}

	imageQueues[ idIt->second][ timestamp].usedCount++;
}

IplImage * TSMManager::getImage( const std::string &queueName, long long timestamp, const bool increaseCounter)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	// check if the queue exists
	const TIDMap::const_iterator idIt = idMap.find( name);
	if( idIt == idMap.end()){
		throw( "TSMManager::getImage(): The requested queue does not exist.");
	}

	const int id = idIt->second;
	const TImageQueue::iterator imgIt = imageQueues[ id].find( timestamp);

	if( imgIt == imageQueues[ id].end()){
		return NULL;
	}

	TUseImage &retImage = imgIt->second;

	// check if the image has been already inserted
	if( retImage.imgPtr == NULL){
		return NULL;
	} 

	IplImage *img = retImage.imgPtr->createIplImage();

	// the image could have been already requested previously by registerImage()
	if( increaseCounter ){ 
		retImage.usedCount++;
	}

	return img;
}

void TSMManager::returnImage( const std::string &queueName, long long timestamp)
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	const TCharString name( queueName.data(), void_alloc);

	const TIDMap::const_iterator idIt = idMap.find( name);
	if( idIt == idMap.end()){
		throw( "TSMManager::returnImage(): The requested queue does not exist.");
	}

	const int id = idIt->second;
	const TImageQueue::iterator imgIt = imageQueues[ id].find( timestamp);

	if( imgIt == imageQueues[ id].end()){
		throw( "TSMManager::returnImage(): The requested timestamp is not present.");
	}

	imgIt->second.usedCount--;

	change();
}


void TSMManager::cleanup()
{
	interprocess::scoped_lock< boost::interprocess::interprocess_mutex> lock(mutex);

	/*	std::cout << "idMap" << idMap.size() << std::endl;
	std::cout << "queueRegisterCounts" << queueRegisterCounts.size() << std::endl;

	for( TImageQues::iterator qIt = imageQueues.begin(); qIt != imageQueues.end(); qIt++){
		std::cout << qIt->size() << std::endl;
	}
	std::cout << std::endl;*/

	for( int id = 0; id < (int) idMap.size(); id++){
	
		TImageQueue::reverse_iterator imgIt = imageQueues[ id].rbegin();

		// we want to keep first two images in each queue
		if( imgIt == imageQueues[ id].rend()) continue;
		long long t1 = imgIt->first;
		imgIt++;
		if( imgIt == imageQueues[ id].rend()) continue;
		long long t2 = imgIt->first;
		imgIt++;

		std::vector< long long> toDelete;

		while( imgIt != imageQueues[ id].rend()){

			TImageQueue::reverse_iterator tempIt = imgIt;
			long long t3 = tempIt->first;
		
			if( tempIt->second.usedCount <= 0){

				if( tempIt->second.imgPtr != NULL){
					void_alloc.get_segment_manager()->destroy_ptr( tempIt->second.imgPtr.get());
				}

				tempIt->second.imgPtr = NULL;

				toDelete.push_back( tempIt->first);
			} 
			imgIt++;
		}

		for( int i = 0; i < (int) toDelete.size(); i++){
			imageQueues[ id].erase( toDelete[ i]);
		}
	}


	for( int i = 0; i < (int) floatQueues.size(); i++){
		floatQueues[ i].erase( floatQueues[ i].begin(), floatQueues[ i].lower_bound( newestTimestamp - floatHistory));
	}

	change();
}

void TSMManager::waitForChange( const int ms)
{
	boost::interprocess::interprocess_mutex tempMutex;

	if( ms <= 0){
		interprocess::scoped_lock< interprocess::interprocess_mutex> lock( tempMutex);
	
		changeCondition.wait( lock);
	} else {

		throw( std::string( "not yet implemented"));
		/*interprocess::scoped_lock< interprocess::interprocess_mutex> lock( *tempMutex);
	
		changeCondition.timed_wait( lock, boost::posix_time::get_system_time() + boost::milliseconds( 10););
*/

	}

}

void TSMManager::change()
{
	changeCondition.notify_all();
}

void TSMManager::test()
{
/*
	if( strVector.empty()){

		cout << "Empty" <<endl;
		strVector.push_back( TCharString( "Jak se sakra m?", void_alloc));
		strVector.push_back( TCharString( "Mm se velmi dobe a jak se m ty?", void_alloc));
		strVector.push_back( TCharString( "J se mm tak dobe, ale tady to stoj za hovno.?", void_alloc));
	} else {
		while( !strVector.empty()){
			cout << strVector.back() << endl;
			strVector.pop_back();
		}
	}
*/


}















